# NestJS Implementation Roadmap - Cart Module (Phase 1)

## Overview
This document provides the detailed implementation plan for building the Cart module in NestJS to match Laravel's shopping cart functionality.

---

## 1. DATABASE SCHEMA

### Cart Table Structure
```sql
CREATE TABLE carts (
  id BIGINT PRIMARY KEY AUTO_INCREMENT,
  user_id BIGINT NOT NULL,
  product_id BIGINT NOT NULL,
  quantity INT NOT NULL DEFAULT 1,
  price DECIMAL(10, 2) NOT NULL,
  variation JSON,
  is_checked BOOLEAN DEFAULT true,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (user_id) REFERENCES users(id),
  FOREIGN KEY (product_id) REFERENCES products(id)
);

CREATE TABLE restock_requests (
  id BIGINT PRIMARY KEY AUTO_INCREMENT,
  user_id BIGINT NOT NULL,
  product_id BIGINT NOT NULL,
  status ENUM('pending', 'approved', 'notified') DEFAULT 'pending',
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (user_id) REFERENCES users(id),
  FOREIGN KEY (product_id) REFERENCES products(id)
);
```

---

## 2. NestJS ENTITY DEFINITIONS

### Cart Entity (`src/modules/cart/entities/cart.entity.ts`)
```typescript
import { Entity, Column, PrimaryGeneratedColumn, ManyToOne, JoinColumn, CreateDateColumn, UpdateDateColumn } from 'typeorm';
import { User } from '../../user/entities/user.entity';
import { Product } from '../../products/entities/product.entity';

@Entity('carts')
export class Cart {
  @PrimaryGeneratedColumn('increment', { type: 'bigint' })
  id: number;

  @Column({ type: 'bigint' })
  user_id: number;

  @Column({ type: 'bigint' })
  product_id: number;

  @Column({ type: 'int', default: 1 })
  quantity: number;

  @Column({ type: 'decimal', precision: 10, scale: 2 })
  price: number;

  @Column({ type: 'json', nullable: true })
  variation: any;

  @Column({ type: 'boolean', default: true })
  is_checked: boolean;

  @CreateDateColumn()
  created_at: Date;

  @UpdateDateColumn()
  updated_at: Date;

  // Relations
  @ManyToOne(() => User, user => user.carts)
  @JoinColumn({ name: 'user_id' })
  user: User;

  @ManyToOne(() => Product, product => product.carts)
  @JoinColumn({ name: 'product_id' })
  product: Product;
}
```

### Restock Request Entity (`src/modules/cart/entities/restock-request.entity.ts`)
```typescript
import { Entity, Column, PrimaryGeneratedColumn, ManyToOne, JoinColumn, CreateDateColumn } from 'typeorm';
import { User } from '../../user/entities/user.entity';
import { Product } from '../../products/entities/product.entity';

@Entity('restock_requests')
export class RestockRequest {
  @PrimaryGeneratedColumn('increment', { type: 'bigint' })
  id: number;

  @Column({ type: 'bigint' })
  user_id: number;

  @Column({ type: 'bigint' })
  product_id: number;

  @Column({
    type: 'enum',
    enum: ['pending', 'approved', 'notified'],
    default: 'pending'
  })
  status: string;

  @CreateDateColumn()
  created_at: Date;

  @ManyToOne(() => User)
  @JoinColumn({ name: 'user_id' })
  user: User;

  @ManyToOne(() => Product)
  @JoinColumn({ name: 'product_id' })
  product: Product;
}
```

---

## 3. DTO DEFINITIONS

### Create Cart DTO (`src/modules/cart/dto/create-cart.dto.ts`)
```typescript
import { IsNumber, IsOptional, IsPositive, IsObject } from 'class-validator';

export class CreateCartDto {
  @IsNumber()
  @IsPositive()
  productId: number;

  @IsNumber()
  @IsPositive()
  quantity: number;

  @IsOptional()
  @IsObject()
  variation?: any;
}
```

### Update Cart DTO (`src/modules/cart/dto/update-cart.dto.ts`)
```typescript
import { IsNumber, IsOptional, IsPositive } from 'class-validator';

export class UpdateCartDto {
  @IsNumber()
  @IsPositive()
  cartId: number;

  @IsNumber()
  @IsPositive()
  quantity: number;
}
```

### Select Cart Items DTO (`src/modules/cart/dto/select-cart-items.dto.ts`)
```typescript
import { IsArray, IsNumber, IsBoolean } from 'class-validator';

export class SelectCartItemDto {
  @IsNumber()
  cartId: number;

  @IsBoolean()
  isSelected: boolean;
}

export class SelectCartItemsDto {
  @IsArray()
  items: SelectCartItemDto[];
}
```

### Restock Request DTO (`src/modules/cart/dto/restock-request.dto.ts`)
```typescript
import { IsNumber, IsPositive } from 'class-validator';

export class RestockRequestDto {
  @IsNumber()
  @IsPositive()
  productId: number;
}
```

---

## 4. SERVICE IMPLEMENTATION

### Cart Service (`src/modules/cart/services/cart.service.ts`)
```typescript
import { Injectable, BadRequestException, NotFoundException } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { Cart } from '../entities/cart.entity';
import { RestockRequest } from '../entities/restock-request.entity';
import { Product } from '../../products/entities/product.entity';
import { CreateCartDto } from '../dto/create-cart.dto';
import { UpdateCartDto } from '../dto/update-cart.dto';
import { RestockRequestDto } from '../dto/restock-request.dto';
import { SelectCartItemsDto } from '../dto/select-cart-items.dto';

@Injectable()
export class CartService {
  constructor(
    @InjectRepository(Cart)
    private cartRepository: Repository<Cart>,
    @InjectRepository(Product)
    private productRepository: Repository<Product>,
    @InjectRepository(RestockRequest)
    private restockRepository: Repository<RestockRequest>,
  ) {}

  /**
   * Get all cart items for a user
   * @param userId User ID
   * @returns Cart items with product details
   */
  async getCart(userId: number) {
    try {
      const cartItems = await this.cartRepository.find({
        where: { user_id: userId },
        relations: ['product'],
      });

      const total = cartItems.reduce((sum, item) => sum + (item.price * item.quantity), 0);

      return {
        success: true,
        data: {
          items: cartItems.map(item => ({
            id: item.id,
            productId: item.product_id,
            productName: item.product.name,
            quantity: item.quantity,
            price: item.price,
            total: item.price * item.quantity,
            variation: item.variation,
            isChecked: item.is_checked,
          })),
          totalItems: cartItems.length,
          checkedItems: cartItems.filter(i => i.is_checked).length,
          cartTotal: total,
        },
      };
    } catch (error) {
      throw new BadRequestException('Failed to fetch cart');
    }
  }

  /**
   * Add product to cart
   * @param userId User ID
   * @param createCartDto Product details
   * @returns Added cart item
   */
  async addToCart(userId: number, createCartDto: CreateCartDto) {
    try {
      // Verify product exists and has stock
      const product = await this.productRepository.findOne({
        where: { id: createCartDto.productId },
      });

      if (!product) {
        throw new NotFoundException('Product not found');
      }

      if (product.current_stock < createCartDto.quantity) {
        throw new BadRequestException('Insufficient stock');
      }

      // Check if product already in cart
      let cartItem = await this.cartRepository.findOne({
        where: {
          user_id: userId,
          product_id: createCartDto.productId,
        },
      });

      if (cartItem) {
        // Update quantity if already in cart
        cartItem.quantity += createCartDto.quantity;
        cartItem = await this.cartRepository.save(cartItem);
      } else {
        // Create new cart item
        const newCartItem = this.cartRepository.create({
          user_id: userId,
          product_id: createCartDto.productId,
          quantity: createCartDto.quantity,
          price: product.unit_price,
          variation: createCartDto.variation || null,
          is_checked: true,
        });
        cartItem = await this.cartRepository.save(newCartItem);
      }

      return {
        success: true,
        message: 'Product added to cart',
        data: {
          id: cartItem.id,
          productId: cartItem.product_id,
          quantity: cartItem.quantity,
          price: cartItem.price,
          total: cartItem.price * cartItem.quantity,
        },
      };
    } catch (error) {
      throw new BadRequestException(error.message || 'Failed to add to cart');
    }
  }

  /**
   * Update cart item quantity
   * @param userId User ID
   * @param updateCartDto Cart update data
   * @returns Updated cart item
   */
  async updateCart(userId: number, updateCartDto: UpdateCartDto) {
    try {
      const cartItem = await this.cartRepository.findOne({
        where: {
          id: updateCartDto.cartId,
          user_id: userId,
        },
        relations: ['product'],
      });

      if (!cartItem) {
        throw new NotFoundException('Cart item not found');
      }

      if (cartItem.product.current_stock < updateCartDto.quantity) {
        throw new BadRequestException('Insufficient stock');
      }

      cartItem.quantity = updateCartDto.quantity;
      const updatedItem = await this.cartRepository.save(cartItem);

      return {
        success: true,
        message: 'Cart updated',
        data: {
          id: updatedItem.id,
          productId: updatedItem.product_id,
          quantity: updatedItem.quantity,
          total: updatedItem.price * updatedItem.quantity,
        },
      };
    } catch (error) {
      throw new BadRequestException(error.message || 'Failed to update cart');
    }
  }

  /**
   * Remove item from cart
   * @param userId User ID
   * @param cartId Cart item ID
   * @returns Success message
   */
  async removeFromCart(userId: number, cartId: number) {
    try {
      const cartItem = await this.cartRepository.findOne({
        where: {
          id: cartId,
          user_id: userId,
        },
      });

      if (!cartItem) {
        throw new NotFoundException('Cart item not found');
      }

      await this.cartRepository.delete(cartId);

      return {
        success: true,
        message: 'Item removed from cart',
      };
    } catch (error) {
      throw new BadRequestException(error.message || 'Failed to remove from cart');
    }
  }

  /**
   * Clear all items from cart
   * @param userId User ID
   * @returns Success message
   */
  async clearCart(userId: number) {
    try {
      await this.cartRepository.delete({ user_id: userId });

      return {
        success: true,
        message: 'Cart cleared',
      };
    } catch (error) {
      throw new BadRequestException('Failed to clear cart');
    }
  }

  /**
   * Update selected items in cart (for checkout)
   * @param userId User ID
   * @param selectCartItemsDto Items to select/deselect
   * @returns Updated cart
   */
  async selectCartItems(userId: number, selectCartItemsDto: SelectCartItemsDto) {
    try {
      for (const item of selectCartItemsDto.items) {
        const cartItem = await this.cartRepository.findOne({
          where: {
            id: item.cartId,
            user_id: userId,
          },
        });

        if (!cartItem) {
          throw new NotFoundException(`Cart item ${item.cartId} not found`);
        }

        cartItem.is_checked = item.isSelected;
        await this.cartRepository.save(cartItem);
      }

      return this.getCart(userId);
    } catch (error) {
      throw new BadRequestException(error.message || 'Failed to select items');
    }
  }

  /**
   * Request product restock notification
   * @param userId User ID
   * @param restockRequestDto Product ID
   * @returns Created restock request
   */
  async requestRestock(userId: number, restockRequestDto: RestockRequestDto) {
    try {
      // Check if product exists
      const product = await this.productRepository.findOne({
        where: { id: restockRequestDto.productId },
      });

      if (!product) {
        throw new NotFoundException('Product not found');
      }

      // Check if already requested
      const existing = await this.restockRepository.findOne({
        where: {
          user_id: userId,
          product_id: restockRequestDto.productId,
          status: 'pending',
        },
      });

      if (existing) {
        return {
          success: false,
          message: 'Restock request already exists',
        };
      }

      const restockRequest = this.restockRepository.create({
        user_id: userId,
        product_id: restockRequestDto.productId,
        status: 'pending',
      });

      const savedRequest = await this.restockRepository.save(restockRequest);

      return {
        success: true,
        message: 'Restock request submitted',
        data: {
          id: savedRequest.id,
          productId: savedRequest.product_id,
          status: savedRequest.status,
        },
      };
    } catch (error) {
      throw new BadRequestException(error.message || 'Failed to request restock');
    }
  }
}
```

---

## 5. CONTROLLER IMPLEMENTATION

### Cart Controller (`src/modules/cart/controllers/cart.controller.ts`)
```typescript
import {
  Controller,
  Get,
  Post,
  Put,
  Delete,
  Body,
  UseGuards,
  Request,
  HttpStatus,
  HttpCode,
  BadRequestException,
} from '@nestjs/common';
import { JwtAuthGuard } from '../../auth/guards/jwt-auth.guard';
import { CartService } from '../services/cart.service';
import { CreateCartDto } from '../dto/create-cart.dto';
import { UpdateCartDto } from '../dto/update-cart.dto';
import { RestockRequestDto } from '../dto/restock-request.dto';
import { SelectCartItemsDto } from '../dto/select-cart-items.dto';

@Controller('api/v1/cart')
export class CartController {
  constructor(private readonly cartService: CartService) {}

  /**
   * GET /api/v1/cart
   * Retrieve all cart items for the current user
   */
  @Get()
  async getCart(@Request() req) {
    // Allow both authenticated and guest users
    const userId = req.user?.id || req.headers['guest-id'];
    
    if (!userId) {
      throw new BadRequestException('User ID or Guest ID required');
    }

    return await this.cartService.getCart(userId);
  }

  /**
   * POST /api/v1/cart/add
   * Add a product to the cart
   */
  @Post('add')
  @HttpCode(HttpStatus.CREATED)
  async addToCart(@Request() req, @Body() createCartDto: CreateCartDto) {
    const userId = req.user?.id || req.headers['guest-id'];
    
    if (!userId) {
      throw new BadRequestException('User ID or Guest ID required');
    }

    return await this.cartService.addToCart(userId, createCartDto);
  }

  /**
   * PUT /api/v1/cart/update
   * Update cart item quantity
   */
  @Put('update')
  async updateCart(@Request() req, @Body() updateCartDto: UpdateCartDto) {
    const userId = req.user?.id || req.headers['guest-id'];
    
    if (!userId) {
      throw new BadRequestException('User ID or Guest ID required');
    }

    return await this.cartService.updateCart(userId, updateCartDto);
  }

  /**
   * DELETE /api/v1/cart/remove
   * Remove an item from the cart
   */
  @Delete('remove')
  async removeFromCart(@Request() req, @Body() body: { cartId: number }) {
    const userId = req.user?.id || req.headers['guest-id'];
    
    if (!userId) {
      throw new BadRequestException('User ID or Guest ID required');
    }

    return await this.cartService.removeFromCart(userId, body.cartId);
  }

  /**
   * DELETE /api/v1/cart/remove-all
   * Clear all items from the cart
   */
  @Delete('remove-all')
  async clearCart(@Request() req) {
    const userId = req.user?.id || req.headers['guest-id'];
    
    if (!userId) {
      throw new BadRequestException('User ID or Guest ID required');
    }

    return await this.cartService.clearCart(userId);
  }

  /**
   * POST /api/v1/cart/select-cart-items
   * Select/deselect items for checkout
   */
  @Post('select-cart-items')
  async selectCartItems(@Request() req, @Body() selectCartItemsDto: SelectCartItemsDto) {
    const userId = req.user?.id || req.headers['guest-id'];
    
    if (!userId) {
      throw new BadRequestException('User ID or Guest ID required');
    }

    return await this.cartService.selectCartItems(userId, selectCartItemsDto);
  }

  /**
   * POST /api/v1/cart/product-restock-request
   * Request product restock notification
   */
  @Post('product-restock-request')
  async requestRestock(@Request() req, @Body() restockRequestDto: RestockRequestDto) {
    const userId = req.user?.id;
    
    if (!userId) {
      throw new BadRequestException('Authentication required');
    }

    return await this.cartService.requestRestock(userId, restockRequestDto);
  }
}
```

---

## 6. MODULE CONFIGURATION

### Cart Module (`src/modules/cart/cart.module.ts`)
```typescript
import { Module } from '@nestjs/common';
import { TypeOrmModule } from '@nestjs/typeorm';
import { Cart } from './entities/cart.entity';
import { RestockRequest } from './entities/restock-request.entity';
import { Product } from '../products/entities/product.entity';
import { CartService } from './services/cart.service';
import { CartController } from './controllers/cart.controller';

@Module({
  imports: [
    TypeOrmModule.forFeature([Cart, RestockRequest, Product]),
  ],
  providers: [CartService],
  controllers: [CartController],
  exports: [CartService],
})
export class CartModule {}
```

---

## 7. INTEGRATION WITH APP MODULE

### Update App Module (`src/app.module.ts`)
```typescript
import { Module } from '@nestjs/common';
import { TypeOrmModule } from '@nestjs/typeorm';
import { ConfigModule } from '@nestjs/config';
import { CartModule } from './modules/cart/cart.module';
// ... other imports

@Module({
  imports: [
    ConfigModule.forRoot(),
    TypeOrmModule.forRoot({
      type: 'mysql',
      host: process.env.DB_HOST || 'localhost',
      port: parseInt(process.env.DB_PORT) || 3306,
      username: process.env.DB_USER || 'root',
      password: process.env.DB_PASSWORD || '',
      database: process.env.DB_NAME || 'multivendor',
      entities: [__dirname + '/**/*.entity{.ts,.js}'],
      synchronize: false,
      logging: process.env.NODE_ENV !== 'production',
    }),
    CartModule,  // Add this
    // ... other modules
  ],
})
export class AppModule {}
```

---

## 8. TESTING STRATEGY

### Test File (`src/modules/cart/cart.service.spec.ts`)
```typescript
import { Test, TestingModule } from '@nestjs/testing';
import { getRepositoryToken } from '@nestjs/typeorm';
import { CartService } from './cart.service';
import { Cart } from './entities/cart.entity';
import { Product } from '../products/entities/product.entity';
import { RestockRequest } from './entities/restock-request.entity';

describe('CartService', () => {
  let service: CartService;
  let mockCartRepository;
  let mockProductRepository;
  let mockRestockRepository;

  beforeEach(async () => {
    mockCartRepository = {
      find: jest.fn(),
      findOne: jest.fn(),
      create: jest.fn(),
      save: jest.fn(),
      delete: jest.fn(),
    };

    mockProductRepository = {
      findOne: jest.fn(),
    };

    mockRestockRepository = {
      findOne: jest.fn(),
      create: jest.fn(),
      save: jest.fn(),
    };

    const module: TestingModule = await Test.createTestingModule({
      providers: [
        CartService,
        {
          provide: getRepositoryToken(Cart),
          useValue: mockCartRepository,
        },
        {
          provide: getRepositoryToken(Product),
          useValue: mockProductRepository,
        },
        {
          provide: getRepositoryToken(RestockRequest),
          useValue: mockRestockRepository,
        },
      ],
    }).compile();

    service = module.get<CartService>(CartService);
  });

  it('should be defined', () => {
    expect(service).toBeDefined();
  });

  describe('addToCart', () => {
    it('should add product to cart', async () => {
      const userId = 1;
      const createCartDto = {
        productId: 1,
        quantity: 2,
      };

      mockProductRepository.findOne.mockResolvedValue({
        id: 1,
        unit_price: 99.99,
        current_stock: 10,
      });

      mockCartRepository.findOne.mockResolvedValue(null);
      mockCartRepository.create.mockReturnValue({
        user_id: userId,
        product_id: 1,
        quantity: 2,
      });
      mockCartRepository.save.mockResolvedValue({
        id: 1,
        user_id: userId,
        product_id: 1,
        quantity: 2,
        price: 99.99,
      });

      const result = await service.addToCart(userId, createCartDto);

      expect(result.success).toBe(true);
      expect(result.message).toBe('Product added to cart');
    });
  });

  describe('getCart', () => {
    it('should return cart items', async () => {
      const userId = 1;

      mockCartRepository.find.mockResolvedValue([
        {
          id: 1,
          user_id: userId,
          product_id: 1,
          quantity: 2,
          price: 99.99,
          product: { name: 'Test Product' },
        },
      ]);

      const result = await service.getCart(userId);

      expect(result.success).toBe(true);
      expect(result.data.items.length).toBe(1);
    });
  });
});
```

---

## 9. CURL TESTING COMMANDS

```bash
# Test 1: Add to Cart
curl -X POST http://localhost:3000/api/v1/cart/add \
  -H "Authorization: Bearer YOUR_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "productId": 1,
    "quantity": 2,
    "variation": {"color": "red", "size": "M"}
  }'

# Test 2: Get Cart
curl -X GET http://localhost:3000/api/v1/cart \
  -H "Authorization: Bearer YOUR_TOKEN"

# Test 3: Update Cart
curl -X PUT http://localhost:3000/api/v1/cart/update \
  -H "Authorization: Bearer YOUR_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "cartId": 1,
    "quantity": 5
  }'

# Test 4: Remove from Cart
curl -X DELETE http://localhost:3000/api/v1/cart/remove \
  -H "Authorization: Bearer YOUR_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "cartId": 1
  }'

# Test 5: Clear Cart
curl -X DELETE http://localhost:3000/api/v1/cart/remove-all \
  -H "Authorization: Bearer YOUR_TOKEN"

# Test 6: Select Cart Items
curl -X POST http://localhost:3000/api/v1/cart/select-cart-items \
  -H "Authorization: Bearer YOUR_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "items": [
      {"cartId": 1, "isSelected": true},
      {"cartId": 2, "isSelected": false}
    ]
  }'

# Test 7: Request Restock
curl -X POST http://localhost:3000/api/v1/cart/product-restock-request \
  -H "Authorization: Bearer YOUR_TOKEN" \
  -H "Content-Type: application/json" \
  -d '{
    "productId": 1
  }'
```

---

## 10. IMPLEMENTATION STEPS

1. **Create Cart Entity**
   ```bash
   cd "EAfricaMall Backend"
   mkdir -p src/modules/cart/entities
   # Create cart.entity.ts with code from section 2
   ```

2. **Create DTOs**
   ```bash
   mkdir -p src/modules/cart/dto
   # Create all DTO files
   ```

3. **Create Service**
   ```bash
   mkdir -p src/modules/cart/services
   # Create cart.service.ts
   ```

4. **Create Controller**
   ```bash
   mkdir -p src/modules/cart/controllers
   # Create cart.controller.ts
   ```

5. **Create Module**
   ```bash
   # Create cart.module.ts
   ```

6. **Update App Module**
   ```bash
   # Add CartModule to imports
   ```

7. **Add Database Migration**
   ```bash
   npm run typeorm migration:create -- src/migrations/CreateCartTables
   ```

8. **Run Migrations**
   ```bash
   npm run typeorm migration:run
   ```

9. **Test Endpoints**
   ```bash
   npm run start:dev
   # Use cURL commands from section 9
   ```

---

## 11. ESTIMATED TIME

- Database Setup: **30 minutes**
- Entity Creation: **15 minutes**
- DTO Creation: **15 minutes**
- Service Implementation: **1 hour**
- Controller Implementation: **30 minutes**
- Module Configuration: **15 minutes**
- Testing: **1 hour**
- **Total: 4 hours**

---

## 12. SUCCESS CRITERIA

✅ All 7 cart endpoints working  
✅ Database entities created  
✅ CRUD operations tested  
✅ Guest user support working  
✅ Error handling implemented  
✅ Response format standardized  
✅ Relationships properly configured  

